using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text.Json;
using System.Threading;
using System.Threading.Tasks;
using DPC.APC.Plugins.SDK;
using Microsoft.Extensions.Logging;
using SupplementaryAreaPlugin.Actions;
using static SupplementaryAreaPlugin.SupplementaryAreasConstants;
public sealed partial class SupplementaryAreasPlugin : IPlugin
{
	private IAppAccess? app;
	internal IAppAccess? App => app; // internal accessor to eliminate reflection

	// --- Lifecycle ---
	public Task InitAsync(IAppAccess app, CancellationToken ct)
	{
		if (app == null) throw new ArgumentNullException(nameof(app));
		this.app = app;
		app.Logger.LogInformation("SupplementaryAreasPlugin initialised");
		return Task.CompletedTask;
	}

	public Task ShutdownAsync(CancellationToken ct)
	{
		app?.Logger.LogInformation("SupplementaryAreasPlugin shutting down");
		return Task.CompletedTask;
	}

	public void Dispose() { }

	// --- Webhook (not used presently) ---
	public Task<bool> ProcessWebhookAsync(string body, int webhookType, CancellationToken ct) => Task.FromResult(false);

	// --- GET Routing ---
	public async Task<PluginResponse> GetAsync(IReadOnlyDictionary<string, string?> query, CancellationToken ct)
	{
		if (app == null) return Error(500, ErrorNotInitialised);
		return await ParseAndHandleGetActionAsync(query, ct);
	}

	private async Task<PluginResponse> ParseAndHandleGetActionAsync(IReadOnlyDictionary<string, string?> query, CancellationToken ct)
	{
		string action = GetQueryValue(query, QueryAction);
		if (string.Equals(action, PeopleSearchAction, StringComparison.OrdinalIgnoreCase))
			return await HandlePeopleSearchAsync(query, ct);
		if (string.Equals(action, "debugModel", StringComparison.OrdinalIgnoreCase))
			return await HandleDebugModelAsync(query, ct);
		return await HandleMainGetActionAsync(query, ct);
	}

	private ActionDebugModel? _debugModel;
	private async Task<PluginResponse> HandleDebugModelAsync(IReadOnlyDictionary<string, string?> query, CancellationToken ct)
	{
		_debugModel ??= new ActionDebugModel(new DebugModelContext(this));
		return await _debugModel.ExecuteAsync(query, ct);
	}

	private string GetQueryValue(IReadOnlyDictionary<string, string?> query, string key)
	   => query.TryGetValue(key, out var value) ? (value ?? string.Empty) : string.Empty;

	private async Task<PluginResponse> HandleMainGetActionAsync(IReadOnlyDictionary<string, string?> query, CancellationToken ct)
	{
		modelRepository ??= new ModelRepository(this);
		getSupplementary ??= new ActionGetSupplementary(new GetSupplementaryContext(this), modelRepository);
		return await getSupplementary.ExecuteAsync(query, ct);
	}

	private bool TryGetRecordId(IReadOnlyDictionary<string, string?> query, out string? recordId)
	{
		recordId = GetQueryValue(query, QueryRecordId);
		return !string.IsNullOrWhiteSpace(recordId);
	}

	// People search
	private ActionPeopleSearch? _peopleSearch;
	private async Task<PluginResponse> HandlePeopleSearchAsync(IReadOnlyDictionary<string, string?> query, CancellationToken ct)
	{
		_peopleSearch ??= new ActionPeopleSearch(new PeopleSearchContext(this));
		return await _peopleSearch.ExecuteAsync(query, ct);
	}

	// --- POST entry point ---
	public async Task<PluginResponse> PostAsync(IReadOnlyDictionary<string, string?> query, string? body, string? contentType, CancellationToken ct)
	{
		if (app == null) return Error(500, ErrorNotInitialised);
		if (!TryGetRecordId(query, out var recordId) || !IsValidRecordId(recordId))
			return Error(400, ErrorMissingRecordId, "Invalid record ID format");
		if (!IsValidJsonPayloadSize(body))
			return Error(400, "payload_too_large", $"JSON payload exceeds maximum size limit of {MaxJsonPayloadSize} characters");

		app.Logger.LogDebug("PostAsync called with recordId={RecordId}, action={Action}", recordId, GetQueryValue(query, QueryAction));
		app.Logger.LogTrace("PostAsync body: {Body}", body);

		var parseResult = ParseJsonBody(body, contentType);
		if (!parseResult.IsValid) return parseResult.ErrorResponse!;
		string action = GetQueryValue(query, QueryAction);
		return await RoutePostActionAsync(action, recordId!, parseResult.JsonRoot, ct);
	}

	private (bool IsValid, PluginResponse? ErrorResponse, JsonElement JsonRoot) ParseJsonBody(string? body, string? contentType)
	{
		if (string.IsNullOrWhiteSpace(body))
			return (true, null, default);
		try
		{
			using var doc = JsonDocument.Parse(body);
			return (true, null, doc.RootElement.Clone());
		}
		catch (Exception ex)
		{
			if (contentType?.StartsWith("application/json", StringComparison.OrdinalIgnoreCase) == true)
				return (false, Error(400, ErrorInvalidJson, ex.Message), default);
			return (false, null, default);
		}
	}

	private async Task<PluginResponse> RoutePostActionAsync(string action, string recordId, JsonElement jsonRoot, CancellationToken ct)
	{
		modelRepository ??= new ModelRepository(this);
		switch (action)
		{
			case UpdateRecordAction:
				updateRecord ??= new ActionUpdateRecord(new UpdateRecordContext(this), modelRepository);
				return await updateRecord.ExecuteAsync(recordId, jsonRoot, ct);
			default:
				return CreateErrorResponse(400, ErrorUnknownAction, $"Unknown action: {action}");
		}
	}

	// --- Validation helpers (will later move to shared ValidationHelpers) ---
	private static bool IsValidJsonPayloadSize(string? body) => SupplementaryAreaPlugin.Shared.ValidationHelpers.IsValidJsonPayloadSize(body, MaxJsonPayloadSize);
	private static bool IsValidStringLength(string? value, int maxLength) => SupplementaryAreaPlugin.Shared.ValidationHelpers.IsValidStringLength(value, maxLength);
	private static bool IsValidRecordId(string? recordId) => SupplementaryAreaPlugin.Shared.ValidationHelpers.IsValidRecordId(recordId);

	// --- Action wiring fields ---
	private ActionGetSupplementary? getSupplementary;
	private IModelRepository? modelRepository;
	private ActionUpdateRecord? updateRecord;

	private PluginResponse CreateErrorResponse(int statusCode, string errorCode, string? detail = null) => Error(statusCode, errorCode, detail);

	private static PluginResponse Json200(object obj) => new()
	{
		StatusCode = 200,
		ContentType = "application/json",
		Body = JsonSerializer.Serialize(obj, JsonOptions)
	};

	private static PluginResponse Error(int status, string code, string? detail = null)
	{
		object payload = detail == null ? new { error = code } : new { error = code, detail };
		return new()
		{
			StatusCode = status,
			ContentType = "application/json",
			Body = JsonSerializer.Serialize<object>(payload, JsonOptions)
		};
	}

	internal static PluginResponse CreateError(int status, string code, string? detail = null) => Error(status, code, detail);

	// --- Context adapters ---
	private sealed class PeopleSearchContext : IPeopleSearchContext
	{
		private readonly SupplementaryAreasPlugin _plugin;
		public PeopleSearchContext(SupplementaryAreasPlugin plugin) => _plugin = plugin;
		public IAppAccess? App => _plugin.app;
		public PluginResponse Ok(object shape) => Json200(shape);
		public PluginResponse Fail(int status, string code, string? detail = null) => Error(status, code, detail);
		public string GetQueryValue(IReadOnlyDictionary<string, string?> query, string key) => _plugin.GetQueryValue(query, key);
		// IMPORTANT: qualify static helper on outer class to avoid recursive self-call causing stack overflow
		public bool IsValidStringLength(string? value, int maxLength) => SupplementaryAreasPlugin.IsValidStringLength(value, maxLength);
	}

	private sealed class DebugModelContext : IDebugModelContext
	{
		private readonly SupplementaryAreasPlugin _plugin;
		public DebugModelContext(SupplementaryAreasPlugin plugin) => _plugin = plugin;
		public IAppAccess? App => _plugin.app;
		public PluginResponse Ok(object shape) => Json200(shape);
		public PluginResponse Fail(int status, string code, string? detail = null) => Error(status, code, detail);
		public string GetQueryValue(IReadOnlyDictionary<string, string?> query, string key) => _plugin.GetQueryValue(query, key);
	}

	private sealed class GetSupplementaryContext : IGetSupplementaryContext
	{
		private readonly SupplementaryAreasPlugin _plugin;
		public GetSupplementaryContext(SupplementaryAreasPlugin plugin) => _plugin = plugin;
		public IAppAccess? App => _plugin.app;
		public PluginResponse Ok(object shape) => Json200(shape);
		public PluginResponse Fail(int status, string code, string? detail = null) => Error(status, code, detail);
		public string GetQueryValue(IReadOnlyDictionary<string, string?> query, string key) => _plugin.GetQueryValue(query, key);
	}

	private sealed class UpdateRecordContext : IUpdateRecordContext
	{
		private readonly SupplementaryAreasPlugin _plugin;
		public UpdateRecordContext(SupplementaryAreasPlugin plugin) => _plugin = plugin;
		public IAppAccess? App => _plugin.app;
		public PluginResponse Ok(object shape) => Json200(shape);
		public PluginResponse Fail(int status, string code, string? detail = null) => Error(status, code, detail);
	}

	private static readonly JsonSerializerOptions JsonOptions = new(JsonSerializerDefaults.Web) { WriteIndented = false };

	private readonly record struct DropdownOption(string id, string label);

	internal sealed class FieldDescriptor
	{
		public string InternalName { get; set; } = string.Empty;
		public string? DisplayName { get; set; }
		public string? Type { get; set; }
	}
}

